

-- Adressen-Kürzel  generieren #10140
CREATE OR REPLACE FUNCTION TAdk.adk__ad_krz__generate(
    IN Firma      VARCHAR,
    IN Nachname   VARCHAR,
    IN Vorname    VARCHAR,
    IN Ort        VARCHAR,
    IN mitOrt     BOOLEAN DEFAULT true,
    IN Term1_len  INTEGER DEFAULT 5,
    IN adkrz      VARCHAR DEFAULT NULL
    ) 
    RETURNS VARCHAR AS $$
    DECLARE resultadkrz VARCHAR;
            firma_parts VARCHAR;
            part_count  INTEGER;
    BEGIN
      -- Ziellänge von Teil "Firma"
      IF COALESCE(Term1_len, 0) <= 0  THEN Term1_len:= 5; -- unsinnige Eingabe auf Default 5
      ELSIF Term1_len > 8             THEN Term1_len:= 8; -- max. Länge 8
      END IF;
  
      -- Daten ggf. aus Adressstamm holen (führend).
      IF adkrz IS NOT NULL AND EXISTS(SELECT true FROM adk WHERE ad_krz = adkrz) THEN
          SELECT ad_fa1, ad_name, ad_vorn, ad_ort INTO Firma, Nachname, Vorname, Ort FROM adk WHERE ad_krz = adkrz;
      END IF;
  
      Firma     := unaccent(upper(trim(Firma)));
      Nachname  := unaccent(upper(trim(Nachname)));
      Vorname   := unaccent(upper(trim(Vorname)));
  
      Firma:= TAdk.adk__ad_krz__entfernen_alle_rechtsform(Firma); -- Rechtsformen entfernen
  
      Firma:= regexp_replace(Firma, E'[^A-Z0-9\\s]', '', 'g'); -- Sonderzeichen (keine Buchstaben, Zahlen, Leerzeichen) löschen
  
      -- Mehrfach innere Leerzeichen entfernen.
      Firma     := nullif(regexp_replace(Firma, E'\\s\\s+', ' ', 'g'), '');
      Nachname  := nullif(regexp_replace(Nachname, E'\\s\\s+', ' ', 'g'), '');
      Vorname   := nullif(regexp_replace(Vorname, E'\\s\\s+', ' ', 'g'), '');
  
      IF mitOrt THEN
          Ort   := unaccent(upper(trim(Ort)));
          Ort   := regexp_replace(Ort, E'[^A-Z0-9\\s]', '', 'g');
          Ort   := nullif(regexp_replace(Ort, E'\\s+', '', 'g'), ''); -- alle Leerzeichen entfernen (St. Gallen => STGALLEN => STG)
      END IF;
  
      IF Firma IS NOT NULL THEN
          resultadkrz:= '';
          part_count := 0;
  
          -- Teile der Firmen-Bezeichnung anfügen. Trenner ist Leerzeichen.
          FOR firma_parts IN SELECT regexp_split_to_table(Firma, ' ') LOOP
              part_count := part_count + 1;
  
              IF ( SELECT count(*) FROM regexp_split_to_table(Firma, ' ') ) = 1 THEN  --- Firmenname besteht nur aus ein Wort
                  resultadkrz := substr(firma_parts, 1, Term1_len);
                  EXIT;
              ELSE -- mehr als 1 Term
                  IF part_count = 1 THEN
                      resultadkrz := substr(firma_parts, 1, 5);                        --- erste 5 Zeichen von ertstes Wort
                  ELSEIF part_count = 2 THEN                                           --- 2. Wort (von weitere Wörter?) nur 2 Zeichen
                      resultadkrz := resultadkrz || ' ' || substr(firma_parts, 1, 2);  --- https://prodat-sql.slack.com/archives/C0117AX8E7P/p1721808837678829?thread_ts=1721808259.269559&cid=C0117AX8E7P
                  END IF;
                  IF length(resultadkrz) >= Term1_len THEN
                      EXIT;
                  END IF;
              END IF;
          END LOOP;
      ELSIF Nachname IS NOT NULL THEN
          resultadkrz := substr(Nachname, 1, 14) || COALESCE(', ' || substr(Vorname, 1, 2), ''); -- laut Eingabemaske
      END IF;
  
      IF mitOrt AND Ort IS NOT NULL AND resultadkrz IS NOT NULL THEN
          resultadkrz := resultadkrz || '-' || substr(Ort, 1, 3);
      END IF;
      RETURN upper( trim( resultadkrz ) );
    END $$ LANGUAGE plpgsql;
--
--- #10132 unaccent - Funktionaktivierung
  DO $$ -- Extension erzeugen.
    BEGIN
      BEGIN
          DROP FUNCTION IF EXISTS unaccent(text);
          CREATE EXTENSION IF NOT EXISTS unaccent;
      EXCEPTION WHEN OTHERS THEN
      END;
    END $$ LANGUAGE plpgsql;

  CREATE OR REPLACE FUNCTION public.unaccent(text) RETURNS TEXT AS
      '$libdir/unaccent', 'unaccent_dict'
    LANGUAGE c STABLE STRICT;
--
CREATE OR REPLACE FUNCTION TAdk.adk__ad_krz__entfernen_rechtsform(IN _text VARCHAR, IN _rechtsform VARCHAR) RETURNS VARCHAR AS $$
  DECLARE result VARCHAR;
  BEGIN
    _rechtsform:= ' ' || upper(trim(_rechtsform)); -- Rechtsform soll führende Leerzeichen enthalten.
    _text:= upper(trim(_text));

    IF right(_text, length(_rechtsform)) = _rechtsform THEN  -- Rechtsform steht am Ende: 'Müller AG'. Nur replace liefert ggf. falsches Ergebnis, z.B entfernen von ' AG' aus 'Müller Agentur'.
        result:= substr(_text, 1, length(_text) - length(_rechtsform));
    ELSE                                                     -- Rechtsform steht in der Mitte: 'Müller AG Dresden'
        result:= replace(_text, _rechtsform || ' ', ' ');
    END IF;

    RETURN result;
  END $$ LANGUAGE plpgsql;

  CREATE OR REPLACE FUNCTION TAdk.adk__ad_krz__entfernen_alle_rechtsform(IN _fname VARCHAR) RETURNS VARCHAR AS $$
    DECLARE result      VARCHAR;
            arr_GesForm VARCHAR[]:= ARRAY['AG', 'GmbH', 'GbR', 'OHG', 'KG', 'UG', 'KGaA', 'e.V.', 'LTD', '& Co.'];
    BEGIN
      result:= upper( _fname );
      FOR i IN 1..array_length(arr_GesForm, 1) LOOP
        result:= (SELECT TAdk.adk__ad_krz__entfernen_rechtsform(result, arr_GesForm[i]));
      END LOOP;

      RETURN result;
    END $$ LANGUAGE plpgsql;
--

-- Adressen umbenennen und zusammenführen
CREATE OR REPLACE FUNCTION TAdk.ChangeAdkrz(
  IN  okrz varchar(21),
  IN  nkrz varchar(21),
  IN  canMerge boolean,
  IN  merge_to_sub_address boolean DEFAULT false,
  IN  _new_ada_krzl varchar DEFAULT ''
  )
  RETURNS void
  AS $$
  DECLARE newAdress       boolean;
          new_sub_address varchar;
  BEGIN
        -- Umbenennen (canMerge = false) oder Zusammenführen (canMerge = true)
        -- als abw. Liefer- und/oder Rechnungsadresse zusammenführen (canMerge = true AND merge_to_sub_address = true)

        IF okrz = nkrz THEN -- Abbruch wenn Kürzel gleich.
            RAISE NOTICE 'Adresskürzel sind gleich - alt: % neu: %', okrz, nkrz;
            RETURN;
        END IF;

        newAdress:= NOT EXISTS(SELECT true FROM adressen_view WHERE ad_krz = nkrz); -- adk und adk_adresses
        _new_ada_krzl := nullif(_new_ada_krzl, '');

        RAISE NOTICE 'TAdk.ChangeAdkrz: alt: % neu: % newadress: %', okrz, nkrz, newadress;

        -- Fehler, wenn Zieladresse existiert und keine Erlaubnis zum Zusammenführen aus Oberfläche.
        IF NOT newAdress AND NOT canMerge THEN
            RAISE EXCEPTION '%', lang_text(10948);
        END IF;

        RAISE NOTICE 'TAdk.ChangeAdkrz: SET LOCAL SESSION AUTHORIZATION';


        RAISE NOTICE 'TAdk.ChangeAdkrz: Diable Triggers A';

        -- Trigger abschalten, um unnötige Prüfungen und Berechnungen in Alt-Daten zu vermeiden.
            PERFORM TSystem.trigger__disable( 'adk1', 'adk1__b_10_iu__a1_knr' );  -- Trigger prüfen a1_knr / a2_knr auf Eindeutigkeit.

            PERFORM TSystem.trigger__disable( 'adk2', 'adk2__b_10_iu__a2_knr' );

            PERFORM TSystem.trigger__disable( 'ldsdok' );       -- (alte) Bestellungen nicht neu durchrechnen, oder weiter kaskadieren.

            PERFORM TSystem.trigger__disable( 'ldsdokdokutxt' );

            PERFORM TSystem.trigger__disable( 'auftg' );         -- (alte) Aufträge nicht neu durchrechnen, oder weiter kaskadieren.

            PERFORM TSystem.trigger__disable( 'auftgdokutxt' );

            PERFORM TSystem.trigger__disable( 'lifsch', 'lifsch__b_iu' );     -- Trigger prüft auf Sperrlager und gültige ME.

            PERFORM TSystem.trigger__disable( 'op2', 'op2__b10_iu' );     -- Trigger prüft auf Sperrung KS und AW. Rechnet Zeiten neu.

            PERFORM TSystem.trigger__disable( 'wendat', 'wendat__a_iu_check_web' );     -- Trigger legt neuen Wareneingangsbericht an wenn noch nicht Vorhanden.

            -- Zurückschreiben geändertes Kürzel in Belegposition unterbinden.
            PERFORM TSystem.trigger__disable( 'eingrechdokument' );
            PERFORM TSystem.trigger__disable( 'belegdokument' );
            PERFORM TSystem.trigger__disable( 'belegpos', 'belegpos__a_50_iud' );   -- Verschlagwortung Dokumente => RecnoKeyword wird sowieso unten umgeschrieben

            PERFORM TSystem.trigger__disable( 'picndoku' );            
        --

        -- Audit-Log abschalten: so tun, als wären wir 'syncro' (u.a. wg. modified-Triggern)
        SET LOCAL SESSION AUTHORIZATION 'syncro'; -- Gilt nur bis Ende der Transaktion (SET LOCAL).
                                                  -- Achtung: ON UPDATE CASCADE wird mit User ausgeführt, der OWNER der Zieltabelle ist (normalerweise postgres oder syncro).

        DELETE FROM adkkrz_log WHERE oldkrz = okrz AND newkrz = nkrz;
        INSERT INTO adkkrz_log (oldkrz, newkrz) VALUES (okrz, nkrz);


        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE adk';

        IF newAdress THEN -- Umbenennen:      Hauptdatensatz umschreiben, kadkadiert insb. nach adk1, adk2, adk_adresses und adressen_keys
            UPDATE adk SET ad_krz= nkrz WHERE ad_krz = okrz;
        ELSE              -- Zusammenführen:  Hauptdatensatz beibehalten, Kreditoren- und Debitorendaten umschreiben, wenn in Zieladresse noch keine vorhanden.
            UPDATE adk1 SET a1_krz= nkrz WHERE a1_krz = okrz AND NOT EXISTS(SELECT true FROM adk1 WHERE a1_krz = nkrz);
            UPDATE adk2 SET a2_krz= nkrz WHERE a2_krz = okrz AND NOT EXISTS(SELECT true FROM adk2 WHERE a2_krz = nkrz);
        END IF;

        -- Trigger wieder aktivieren.
            PERFORM TSystem.trigger__enable( 'adk1', 'adk1__b_10_iu__a1_knr' );
            PERFORM TSystem.trigger__enable( 'adk2', 'adk2__b_10_iu__a2_knr' );
        --

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE adk_adresses';

        -- Kürzel der abweichenden Liefer- und Rechnungsadressen auf neues Adresskürzel umschreiben.
        IF newAdress THEN -- Umbenennen
            UPDATE adk_adresses SET ada_krzl = NULL WHERE ada_ad_krz = nkrz; -- ada_krzl neu per adk_adresses__b_iu. ada_ad_krz kaskadiert schon aus Umschreiben von adk.
            -- adressen_keys werden automatisch von adk__a_iud umgeschrieben.
        ELSE              -- Zusammenführen
            UPDATE adk_adresses 
               SET ada_ad_krz = nkrz,
                   ada_pos    = NULL, -- neu per adk_adresses__b_iu
                   ada_krzl   = NULL  -- neu per adk_adresses__b_iu, ada_krzl UNIQUE
             WHERE adk_adresses.ada_ad_krz = okrz;

            -- adressen_keys umhängen an neue Adresse.
            UPDATE adressen_keys SET ak_ad_krz = nkrz
             WHERE ak_ad_krz = okrz AND ak_krz <> okrz;
        END IF;

        -- Adresse als abw. Liefer- oder Rechnungsadresse in Ziel-Adresse mergen.
        -- Wird unten per COALESCE(new_sub_address, nkrz) in entspr. Feldern umgeschrieben.
        IF NOT newAdress AND canMerge AND merge_to_sub_address THEN
            INSERT INTO adk_adresses(
                        ada_ad_krz, ada_krzl, ada_fa1, ada_fa2, ada_adrzus, ada_name, ada_vorn, ada_pfc, ada_str, ada_plz, ada_ort, ada_landiso, ada_land, ada_lgps, ada_bgps, ada_ldauer, ada_auslauf,
                        ada_ladress, 
                        ada_radress,
                        ada_bem
                        )
                 SELECT nkrz, _new_ada_krzl, ad_fa1,  ad_fa2,  ad_adrzus,  ad_name,  ad_vorn,  ad_pfc,  ad_str,  ad_plz,  ad_ort,  ad_landiso,  ad_land,  ad_lgps,  ad_bgps,  ad_ldauer,  ad_auslauf,
                        coalesce(_new_ada_krzl LIKE '%~L-%', true),
                        coalesce(_new_ada_krzl LIKE '%~F-%', false),
                        okrz
                   FROM adk
                  WHERE ad_krz = okrz
              RETURNING ada_krzl INTO new_sub_address; -- neu per adk_adresses__b_iu, ada_krzl UNIQUE

            UPDATE adk SET ad_auslauf = current_date, ad_ersatzadkrz = new_sub_address WHERE ad_krz = okrz;
        ELSE
            new_sub_address:= NULL;
        END IF;

        -- AP der Adresse umschreiben und AP-Kürzel ggf. neu vergeben.
        UPDATE adkap  
           SET ap_ad_krz = nkrz,
               ap_krzl   = CASE WHEN -- UNIQUE INDEX adkap__ap_ad_krz__ap_krzl ON adkap (ap_ad_krz, ap_krzl) WHERE ap_krzl IS NOT NULL
                                  EXISTS(SELECT true FROM adkap AS adkap_trg WHERE adkap_trg.ap_ad_krz = nkrz AND adkap_trg.ap_krzl = adkap.ap_krzl)
                                THEN NULL -- neu per adkap__b_iu
                                ELSE ap_krzl -- bleibt so
                           END
         WHERE ap_ad_krz = okrz;

        -- Zuständige MA für Adresse
        -- vermeide Konflikte mit UNIQUE INDEX adkZust_krz_llminr ON adkZust(azust_ad_krz, azust_ll_db_usename)
        -- nur Update auf alle alten Sätze, die nicht schon in den neuen sind.
        UPDATE adkzust
           SET azust_ad_krz= nkrz
         WHERE azust_ad_krz = okrz
           AND azust_ll_db_usename NOT IN (
                  -- hole alle Mitnr. die schon in dem neuen Satz sind
                  SELECT adkz2.azust_ll_db_usename
                    FROM adkzust AS adkz2
                   WHERE adkz2.azust_ad_krz = nkrz
               );

        -- Tabelle ist kundespezifisch und wird nicht überall erstellt, siehe z50-customer.sql.
        IF EXISTS(SELECT true FROM pg_catalog.pg_tables WHERE schemaname = 'public' AND tablename  = 'airbusk') THEN

            UPDATE airbusk SET
              ai_kun = nkrz
            WHERE ai_kun = okrz
            ;

        END IF;

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE ab2';

        UPDATE ksv                    SET ks_krzl= nkrz                                       WHERE ks_krzl = okrz;
        UPDATE ab2                    SET a2_adkrz= nkrz                                      WHERE a2_adkrz = okrz;

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE art';

        UPDATE art                    SET ak_kunde= nkrz                                      WHERE ak_kunde = okrz;
        UPDATE art                    SET ak_hersteller= nkrz                                 WHERE ak_hersteller = okrz;
        UPDATE art                    SET ak_lerkl_ad_krz= nkrz                               WHERE ak_lerkl_ad_krz = okrz;
        UPDATE artpr                  SET pr_hkrz= nkrz                                       WHERE pr_hkrz = okrz;
        UPDATE artzuo                 SET az_prokrz= nkrz                                     WHERE az_prokrz = okrz;
        UPDATE artikelprognose        SET apr_ad_krz= nkrz                                    WHERE apr_ad_krz = okrz;

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE anl';

        UPDATE anl                    SET an_best= COALESCE(new_sub_address, nkrz)            WHERE an_best = okrz;     -- abw. L&R-Adressen kaskadieren vorher
        UPDATE anl                    SET an_hest= COALESCE(new_sub_address, nkrz)            WHERE an_hest = okrz;     -- abw. L&R-Adressen kaskadieren vorher
        UPDATE anl                    SET an_lief= COALESCE(new_sub_address, nkrz)            WHERE an_lief = okrz;     -- abw. L&R-Adressen kaskadieren vorher
        UPDATE anl                    SET an_inst= COALESCE(new_sub_address, nkrz)            WHERE an_inst = okrz;     -- abw. L&R-Adressen kaskadieren vorher
        UPDATE anl                    SET an_sortkrz= COALESCE(new_sub_address, nkrz)         WHERE an_sortkrz = okrz;  -- abw. L&R-Adressen kaskadieren vorher
        UPDATE anl                    SET an_kund= nkrz                                       WHERE an_kund = okrz;
        UPDATE anl_personen           SET anp_ad_krz= nkrz                                    WHERE anp_ad_krz = okrz;

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE auftg';

        UPDATE auftg                  SET ag_lkn= nkrz                                        WHERE ag_lkn = okrz;
        UPDATE auftg                  SET ag_krzl= COALESCE(new_sub_address, nkrz)            WHERE ag_krzl = okrz;         -- abw. L&R-Adressen kaskadieren vorher
        UPDATE auftg                  SET ag_krzf= COALESCE(new_sub_address, nkrz)            WHERE ag_krzf = okrz;         -- abw. L&R-Adressen kaskadieren vorher
        UPDATE auftgdokutxt           SET atd_versandort= COALESCE(new_sub_address, nkrz)     WHERE atd_versandort = okrz;  -- abw. L&R-Adressen kaskadieren vorher

        UPDATE kundanfrage            SET kanf_ad_krz= nkrz                                   WHERE kanf_ad_krz = okrz;
        UPDATE kundanfrage            SET kanf_verm= nkrz                                     WHERE kanf_verm = okrz;
        UPDATE kundanfrage            SET kanf_krzl= COALESCE(new_sub_address, nkrz)          WHERE kanf_krzl = okrz; -- abw. L&R-Adressen kaskadieren vorher

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE belkopf';

        UPDATE belkopf                SET be_rkrz= COALESCE(new_sub_address, nkrz)            WHERE be_rkrz = okrz;     -- abw. L&R-Adressen kaskadieren vorher
        UPDATE belkopf                SET be_liefkrz= COALESCE(new_sub_address, nkrz)         WHERE be_liefkrz = okrz;  -- abw. L&R-Adressen kaskadieren vorher
        UPDATE belkopf                SET be_sortkrz= COALESCE(new_sub_address, nkrz)         WHERE be_sortkrz = okrz;  -- abw. L&R-Adressen kaskadieren vorher

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE EINKAUF';

        UPDATE bestanfpos             SET bap_lkn= nkrz                                       WHERE bap_lkn = okrz AND (bap_lkn IS NOT NULL);
        UPDATE bestvorschlagpos       SET bvp_lkn= nkrz                                       WHERE bvp_lkn = okrz;
        UPDATE epreis                 SET e_lkn= nkrz                                         WHERE e_lkn = okrz;
        UPDATE epreiskat              SET ek_ad_krz = nkrz                                    WHERE ek_ad_krz = okrz;
        UPDATE eprzutxt               SET ezt_lkn= nkrz                                       WHERE ezt_lkn = okrz;
        UPDATE anflief                SET alief_lkn= nkrz                                     WHERE alief_lkn = okrz;

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE ldsdok';

        UPDATE ldsdok                 SET ld_kn= nkrz                                         WHERE ld_kn = okrz;
        UPDATE ldsdok                 SET ld_krzl= coalesce(new_sub_address, nkrz)            WHERE ld_krzl = okrz;         -- abw. L&R-Adressen kaskadieren vorher
        UPDATE ldsdok                 SET ld_krzf= coalesce(new_sub_address, nkrz)            WHERE ld_krzf = okrz;         -- abw. L&R-Adressen kaskadieren vorher
        UPDATE ldsdokdokutxt          SET ltd_versandort= coalesce(new_sub_address, nkrz)     WHERE ltd_versandort = okrz;  -- abw. L&R-Adressen kaskadieren vorher
        UPDATE rahmenlieferant        SET rhl_krz= nkrz                                       WHERE rhl_krz = okrz;

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE LAGER';

        UPDATE invrech                SET ir_ad_krz= nkrz                                     WHERE ir_ad_krz = okrz;

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE lifsch';

        UPDATE lifsch                 SET l_krz= nkrz                                         WHERE l_krz = okrz;
        UPDATE lifsch                 SET l_krzl= coalesce(new_sub_address, nkrz)             WHERE l_krzl = okrz;  -- abw. L&R-Adressen kaskadieren vorher
        UPDATE lifsch                 SET l_krzf= coalesce(new_sub_address, nkrz)             WHERE l_krzf = okrz;  -- abw. L&R-Adressen kaskadieren vorher

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE wendat';

        UPDATE wareneingangskontrolle SET wek_l_krz= nkrz                                     WHERE wek_l_krz=okrz;
        UPDATE wendat                 SET w_l_krz= nkrz                                       WHERE w_l_krz = okrz AND w_l_krz IS NOT NULL;

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE belegdokument';

        UPDATE belegdokument          SET beld_krzbesteller= coalesce(new_sub_address, nkrz)  WHERE beld_krzbesteller = okrz; -- abw. L&R-Adressen kaskadieren vorher
        UPDATE belegdokument          SET beld_krzlieferung= coalesce(new_sub_address, nkrz)  WHERE beld_krzlieferung = okrz; -- abw. L&R-Adressen kaskadieren vorher
        UPDATE belegdokument          SET beld_krzrechnung= coalesce(new_sub_address, nkrz)   WHERE beld_krzrechnung = okrz;  -- abw. L&R-Adressen kaskadieren vorher
        UPDATE belegdokument          SET beld_versandort= coalesce(new_sub_address, nkrz)    WHERE beld_versandort = okrz;   -- abw. L&R-Adressen kaskadieren vorher

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE belegpos';

        UPDATE belegpos               SET belp_krzbesteller= coalesce(new_sub_address, nkrz)  WHERE belp_krzbesteller = okrz; -- abw. L&R-Adressen kaskadieren vorher
        UPDATE belegpos               SET belp_krzrechnung= coalesce(new_sub_address, nkrz)   WHERE belp_krzrechnung = okrz;  -- abw. L&R-Adressen kaskadieren vorher
        UPDATE belegpos               SET belp_krzlieferung= coalesce(new_sub_address, nkrz)  WHERE belp_krzlieferung = okrz; -- abw. L&R-Adressen kaskadieren vorher


        UPDATE nkz                    SET nz_ad_krz= nkrz                                     WHERE nz_ad_krz = okrz;
        UPDATE op2                    SET o2_adkrz= nkrz                                      WHERE o2_adkrz = okrz;
        UPDATE qab                    SET q_krzl= nkrz                                        WHERE q_krzl = okrz;
        UPDATE qab                    SET q_frei1= nkrz                                       WHERE q_frei1 = okrz;

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE DMS: recnokeyword';

        -- UPDATE recnokeyword           SET r_descr = nkrz                                      WHERE r_descr = okrz AND r_kategorie = 'adk'; -- DMS-Schlüsselworte
        IF coalesce(new_sub_address, nkrz) IS DISTINCT FROM okrz THEN
          INSERT INTO recnokeyword 
                      (r_descr, r_kategorie, r_tablename, r_dbrid)
               SELECT coalesce(new_sub_address, nkrz), r_kategorie, r_tablename, r_dbrid
                 FROM recnokeyword
                WHERE r_descr = okrz AND r_kategorie = 'adk';
        END IF;

        UPDATE picndoku SET pd_dbrid = adkn.dbrid FROM adk adkn, adk adko WHERE adko.ad_krz = okrz AND pd_dbrid = adko.dbrid AND adkn.ad_krz = nkrz AND adkn.dbrid <> adko.dbrid;

        -- Notizen!
        UPDATE recnocomments SET rc_dbrid  = adkn.dbrid FROM adk adkn, adk adko WHERE adko.ad_krz = okrz AND rc_dbrid  = adko.dbrid AND adkn.ad_krz = nkrz AND adkn.dbrid <> adko.dbrid;

        RAISE NOTICE 'TAdk.ChangeAdkrz: UPDATE others';

        UPDATE vertrag                SET vtr_krz= nkrz                                       WHERE vtr_krz = okrz;
        UPDATE vertrag_pos            SET vtp_address= coalesce(new_sub_address, nkrz)        WHERE vtp_address = okrz;

        UPDATE NormZert               SET noz_ad_krz= nkrz                                    WHERE noz_ad_krz = okrz;

        UPDATE llv                    SET ll_ad_krz= nkrz                                     WHERE ll_ad_krz = okrz;

        UPDATE personal               SET pers_krz= nkrz                                      WHERE pers_krz = okrz;
        UPDATE personal               SET pers_kv= nkrz                                       WHERE pers_kv = okrz;
        UPDATE personal               SET pers_letztearbeit= nkrz                             WHERE pers_letztearbeit = okrz;
        UPDATE skillplan              SET skp_lkn= nkrz                                       WHERE skp_lkn = okrz;
        UPDATE skillmitarbzu          SET smz_lkn= nkrz                                       WHERE smz_lkn = okrz;
        UPDATE sepaTransaktion        SET sepa_adkrz= nkrz                                    WHERE sepa_adkrz = okrz;

        -- ausgelassen:
            -- UPDATE adklog SET adkl_ad_krz -- um Log nicht durcheinander zu bringen. Wird mit Adresse ggf. gelöscht.
            -- UPDATE adkbewarab SET adwarab_ad_krz -- um Rabatteinstellungen nicht durcheinander zu bringen. Wird mit Adresse ggf. gelöscht.
        --

        RESET SESSION AUTHORIZATION;

        -- Trigger wieder aktivieren.
            PERFORM TSystem.trigger__enable( 'belegdokument' );
            PERFORM TSystem.trigger__enable( 'eingrechdokument' );
            PERFORM TSystem.trigger__enable( 'belegpos' );
            PERFORM TSystem.trigger__enable( 'op2' );
            PERFORM TSystem.trigger__enable( 'lifsch' );
            PERFORM TSystem.trigger__enable( 'auftgdokutxt' );
            PERFORM TSystem.trigger__enable( 'auftg' );
            PERFORM TSystem.trigger__enable( 'ldsdokdokutxt' );
            PERFORM TSystem.trigger__enable( 'ldsdok' );
            PERFORM TSystem.trigger__enable( 'wendat' );

            PERFORM TSystem.trigger__enable( 'picndoku' );
        --

        RETURN;
  END $$ LANGUAGE plpgsql VOLATILE STRICT;
--

--- #16871
CREATE OR REPLACE FUNCTION TAdk.ChangeAdkrz__db_link(
    IN okrz                 varchar,
    IN nkrz                 varchar,
    IN canMerge             boolean,
    IN merge_to_sub_address boolean = false,
    IN _new_ada_krzl varchar DEFAULT null
    ) 
    RETURNS void AS $$
    BEGIN
  
      PERFORM * FROM dblink(tsystem.dblink__connectionstring__get(),
                            'SELECT * FROM TAdk.ChangeAdkrz(' || quote_literal(okrz) || ', ' || quote_literal(nkrz) || ', ' || canMerge::TEXT || ', ' || merge_to_sub_address::TEXT || coalesce(', ' || quote_literal(_new_ada_krzl), '') || ')'
                           ) AS sub (result VARCHAR);
  
      RETURN;
    END $$ LANGUAGE plpgsql;

-- Prüft alle Zertifikate und Normen eines Lieferant auf Gültigkeit
CREATE OR REPLACE FUNCTION TAdk.Check_EKZert_Full(IN adkrz VARCHAR(21),IN CheckDate DATE DEFAULT current_date, OUT normkrz VARCHAR, OUT isValid BOOLEAN, OUT isWarning BOOLEAN, OUT infoText VARCHAR ) RETURNS SETOF RECORD AS $$
 DECLARE r  RECORD;
         rr RECORD;
 BEGIN
  --Durch alle Zerts für den Lieferant laufen und rausgeben was die Prüfung dafür ergeben hat
  FOR r IN SELECT * FROM normzert WHERE noz_ad_krz=adkrz AND noz_iskred LOOP
    SELECT (TAdk.Check_EkZert(adkrz, r.noz_qs_ident,CheckDate)).* INTO rr;
    normkrz:=r.noz_qs_ident;
    isValid:=rr.isValid;
    isWarning:=rr.isWarning;
    infoText:=rr.InfoText;
    RETURN NEXT;
  END LOOP;
  RETURN;
 END $$ LANGUAGE plpgsql STABLE;
--

-- Prüft ein bestimmtes Zertifikat oder Norm eines Lieferant auf Gültigkeit
CREATE OR REPLACE FUNCTION TAdk.Check_EKZert(IN adkrz VARCHAR(21), IN normkrz VARCHAR, IN CheckDate DATE DEFAULT current_date, OUT isValid BOOLEAN, OUT isWarning BOOLEAN, OUT infoText VARCHAR ) RETURNS RECORD AS $$
 DECLARE r RECORD;
 BEGIN

  IF COALESCE(normKrz,'') = '' THEN --Da ist nichts zu prüfen, das wird also schon so stimmen.
    isValid:=true;
    RETURN;
  END IF;

  SELECT * INTO r FROM NormZert WHERE noz_ad_krz = adkrz AND noz_qs_ident = normkrz AND noz_iskred;

  IF r IS NULL THEN --Nichts getroffen? Dann darf der Lieferant das definitiv nicht liefern.
    isValid:=false;
    isWarning:=false; -- Das ist ein Fehler.
    infoText:=TSystem.FormatS( FormatLines( lang_text(13179),           -- Lieferant ist nicht zertifiziert.
                                            lang_text(13525) || ': %',  -- Norm / Zertifikat: xyz
                                            lang_text(525)   || ': %'), -- Lieferant: abc
                                            normkrz, adkrz);
    RETURN;
  END IF;

  -- Noch nicht gültig
  IF r.noz_bdat IS NOT NULL AND (r.noz_bdat > CheckDate) THEN
    isValid:=false;
    isWarning:=NOT r.noz_sperr;
    infoText:=TSystem.FormatS( FormatLines( lang_text(13176),           -- Zertifikat ist noch nicht gültig.
                                            lang_text(13525) || ': %',  -- Norm / Zertifikat: xyz
                                            lang_text(12762) || ': %',  -- Gültig ab: 01-01-1970
                                            lang_text(525)   || ': %'), -- Lieferant: abc
                                            normkrz, r.noz_bdat::VARCHAR, adkrz);
    If r.noz_sperr THEN
      infoText:= FormatLines(infoText, lang_text(13178)); --Vorgang wird abgebrochen.
    END IF;
    RETURN;
  END IF;

  --Nicht mehr gültig
  IF r.noz_edat IS NOT NULL AND (r.noz_edat < CheckDate ) THEN
    isValid:=false;
    isWarning:=NOT r.noz_sperr;
    infoText:=TSystem.FormatS( FormatLines( lang_text(13177),           -- Zertifikat ist nicht mehr gültig.
                                            lang_text(13525) || ': %',  -- Norm / Zertifikat: xyz
                                            lang_text(12611) || ': %',  -- Gültig bis: 01-01-1970
                                            lang_text(525)   || ': %'), -- Lieferant: abc
                                            normkrz, r.noz_edat::VARCHAR, adkrz);
    If r.noz_sperr THEN
      infoText:= FormatLines(infoText, lang_text(13178)); --Vorgang wird abgebrochen.
    END IF;
    RETURN;
  END IF;

  isValid:=true;    -- Scheinbar ok.
  RETURN;
 END $$ LANGUAGE plpgsql STABLE;
--


--Konto und BLZ errechnen aus IBAN errechnen. (Nur DE)
CREATE OR REPLACE FUNCTION TAdk.ktnrblz_from_iban(IN iban VARCHAR(22), OUT ktnr VARCHAR(10), OUT blz VARCHAR(8)) RETURNS RECORD AS $$
 BEGIN
  IF (LENGTH(iban) <> 22) THEN
    blz:=lang_text(977);
    ktnr:=lang_text(977);
  ELSE
    blz:= substr(iban, 5, 8);
    ktnr := substr(iban, 13, 10);
  END IF;
 END $$ LANGUAGE 'plpgsql';
--

--IBAN aus Konto und BLZ errechnen. (Nur DE)
CREATE OR REPLACE FUNCTION TAdk.Iban_from_ktnrblz(ktnr VARCHAR(10), blz VARCHAR(8)) RETURNS VARCHAR AS $$
 DECLARE iban VARCHAR(22);
        num NUMERIC(24,0);
        _ktnr VARCHAR(10);
        _blz  VARCHAR(8);
 BEGIN
  -- Bankleitzahl 70090100 Kontonummer  1234567890  => Iban: DE08700901001234567890
  -- NACH: http://www.iban.de/iban-pruefsumme.html
  IF (LENGTH(ktnr) > 10) OR (LENGTH(blz) > 8) THEN
    return lang_text(977); --"Ungueltig"
  END IF;

  _ktnr:=lpad(ktnr, 10, '0');
  _blz:= lpad(blz,8,'0');

  -- 8 BLZ + 10 Kto + 4 Stellen Laenderkennung + 2 Nullen
  num:= CAST( (_blz || _ktnr || '131400') AS NUMERIC(24,0));
  num:= 98 - mod(num,97);
  iban:='DE' || lpad( CAST(num AS VARCHAR),2,'0') || _blz || _ktnr;
  RETURN iban;
 END $$ LANGUAGE 'plpgsql';
--

--Validiert eine Iban per Checksummenpruefung. Gibt true zurueck wenn gueltig.
-- Kommentar MF: keine alfanumerische Zeichen am Ende möglich..nicht für alle Schweizer IBAN anwendbar
-- vor manuellem Basteln hier ggfs: https://github.com/josepmartorell/IBAN_Checker
CREATE OR REPLACE FUNCTION TAdk.Validate_iban(IN iban VARCHAR(22)) RETURNS BOOLEAN AS $$
 DECLARE
 BEGIN
  Return (1 = mod( CAST( substring(iban,5,18) || '1314' || substring(iban,3,2) AS NUMERIC(24,0)),97 )) ;
 END $$ LANGUAGE 'plpgsql';
--

-- Formatierung einer IBAN mit Leerzeichen nach 4 Stellen
-- lässt alfanumerische Zeichen am Ende zu, Schweizer IBAN
CREATE OR REPLACE FUNCTION TAdk.IBAN__format_block(IN _iban varchar(22))
    RETURNS varchar
    AS $$
        SELECT trim(
                   concat(
                       substring(replace(_iban, ' ', '') FROM 1 FOR 4), ' ',
                       substring(replace(_iban, ' ', '') FROM 5 FOR 4), ' ',
                       substring(replace(_iban, ' ', '') FROM 9 FOR 4), ' ',
                       substring(replace(_iban, ' ', '') FROM 13 FOR 4), ' ',
                       substring(replace(_iban, ' ', '') FROM 17 FOR 4), ' ',
                       substring(replace(_iban, ' ', '') FROM 21 FOR 4), ' ',
                       substring(replace(_iban, ' ', '') FROM 25 FOR 4), ' ',
                       substring(replace(_iban, ' ', '') FROM 29 FOR 4)
                         )
                   ) AS iban;
    $$ LANGUAGE sql STABLE;


-- Prüft über das Länderkürzel, ob eine Überweisung per Sepa prinzipiell möglich ist
CREATE OR REPLACE FUNCTION TAdk.Supports_Sepa(IN landIso VARCHAR) RETURNS BOOLEAN AS $$
 BEGIN
   -- http://www.dpma.de/service/dasdpmainformiert/gebuehrenzahlungbeimdpma/ueberweisung/sepalaender/
   RETURN landiso = ANY ( string_to_array('BE,BG,DK,DE,EE,FI,FR,GR,GB,IE,IS,IT,HR,LV,LI,LT,LU,MT,MC,NL,NO,AT,PL,PT,RO,SE,CH,SK,SI,ES,CZ,HU,CY',','));
 END $$ LANGUAGE plpgsql IMMUTABLE STRICT;
-- Beispiel: SELECT *, TADk.Supports_Sepa(l_iso) FROM laender

-- Erstellt ein neues Ansprechpartner-Kürzel
CREATE OR REPLACE FUNCTION TAdk.CreateAPKrzl(adkrz VARCHAR, apname VARCHAR, apvorn VARCHAR) RETURNS VARCHAR AS $$
  DECLARE exist   BOOLEAN;
          krz     VARCHAR;
          i       INTEGER;
  BEGIN
    -- 1. Buchstabe Nachname + 1. Buchstabe Vorname
    krz:= COALESCE( UPPER(SUBSTRING(apname FROM 1 FOR 1)), '') || COALESCE( UPPER(SUBSTRING(apvorn FROM 1 FOR 1)), '');

    IF COALESCE(krz, '') = '' OR COALESCE(adkrz, '') = '' THEN -- Ohne Namen sind wir hier raus.
        RETURN krz;
    END IF;

    -- Prüfen, ob es dieses AP-Kürzel schon gibt.
    exist:= EXISTS(SELECT true FROM adkap WHERE ap_krzl = krz AND ap_ad_krz = adkrz);

    i:= 1;
    WHILE exist LOOP
        i:= i+1;
        exist:= EXISTS(SELECT true FROM adkap WHERE ap_krzl = krz || (i::VARCHAR) AND ap_ad_krz = adkrz); -- eine Zahl anhängen und kucken ob es auch schon gibt.
    END LOOP;

    -- Kürzel gab es schon, wir haben also hochgezählt
    IF i > 1 THEN
        krz:= krz || (i::VARCHAR);
    END IF;

    RETURN krz;
  END $$ LANGUAGE plpgsql STABLE;
--




----- #8670 Anfang
--- PersonenFunktion. Liste-Erweiterung soll hier sein.  #8670
CREATE OR REPLACE FUNCTION TAdk.AP_SysFun_Defaults(OUT sf_id INTEGER, OUT sf_bez VARCHAR) RETURNS SETOF RECORD AS $$
 BEGIN
   sf_id = 1; sf_bez = prodat_languages.lang_text(29137); -- Geschäftsführung
   RETURN NEXT;
   sf_id = 2; sf_bez = prodat_languages.lang_text(29138); -- Rechnungsempfang
   RETURN NEXT;
   sf_id = 3; sf_bez = prodat_languages.lang_text(25460); -- Empfang Aufgtragsbest.
   RETURN NEXT;
   sf_id = 4; sf_bez = prodat_languages.lang_text(25556); -- Empfangsbest.
   RETURN NEXT;
   sf_id = 5; sf_bez = prodat_languages.lang_text(25636); -- Auswärtsvergaben einheitliche E-Mail Adresse nur # 
   RETURN NEXT;
   RETURN;
 END $$ LANGUAGE plpgsql STABLE;



--- Dokumentversandliste. Liste-Erweiterung soll hier sein.  #8670
CREATE OR REPLACE FUNCTION TAdk.DokVersandTypen_Defaults( OUT ad_dokversand INTEGER, OUT ad_DokVersandBez VARCHAR) RETURNS SETOF RECORD AS $$
 BEGIN
   ad_dokversand = 1; ad_DokVersandBez = prodat_languages.lang_text(29141);   -- Postversand
   RETURN NEXT;
   ad_dokversand = 2; ad_DokVersandBez = prodat_languages.lang_text(6528);       -- E-Mail
   RETURN NEXT;
   ad_dokversand = 3; ad_DokVersandBez = prodat_languages.lang_text(25480);   -- Online-Portal --#12498
   RETURN NEXT;
   RETURN;
 END $$ LANGUAGE plpgsql STABLE;


--- Liefert Dokumentversandtyp von Adressekurz
CREATE OR REPLACE FUNCTION TAdk.adk_DokVersandTyp(IN _ad_krz VARCHAR) RETURNS VARCHAR AS $$
 BEGIN
   RETURN (SELECT (TAdk.DokVersandTypen(ad_dokversand)).ad_DokVersandBez FROM adk WHERE ad_krz = _ad_krz);
 END $$ LANGUAGE plpgsql STABLE;
----- #8670 Ende

-- Gefahrenübergangsort #7990, #12892
CREATE OR REPLACE FUNCTION TAdk.DokVersandOrt(
      _source     varchar,  -- Einkauf = E; Verkauf = V
      _dokunr     integer,  -- Dokumentnummer -> holen der Lieferadresse
      _versandart varchar   -- z.B. EXW
  ) RETURNS varchar AS $$
  BEGIN

      -- Einkauf
      IF _source = 'E' THEN

          -- Lieferant
          IF _versandart IN ( 'EXW', 'FCA' ) THEN

              RETURN ld_kn
                FROM ldsdok
              WHERE ld_dokunr = _dokunr
              LIMIT 1;
          END IF;

          -- Lieferadresse
          IF _versandart IN ( 'frei Haus', 'DAP', 'DDP', 'CPT' ) THEN

              RETURN ld_krzl
                FROM ldsdok
              WHERE ld_dokunr = _dokunr
              LIMIT 1;
          END IF;

      END IF;

      -- Verkauf
      IF _source = 'V' THEN

          -- Eigene Adresse
          IF _versandart IN ( 'EXW', 'FCA' ) THEN
              RETURN coalesce( nullif( TSystem.Settings__Get( 'DokVersandOrt_V' ), '' ), '#' );
          END IF;

          -- Lieferadresse
          IF _versandart IN ( 'DAP', 'DDP' ) THEN

              RETURN ag_krzl
                FROM auftg
              WHERE ag_dokunr = _dokunr
              LIMIT 1;
          END IF;

      END IF;

      -- Wenn Eingangsparameter falsch;
      RETURN null;
  END $$ LANGUAGE plpgsql STABLE STRICT;
--
-- #21806 Anwendungsbeispiel: IBAN Validierung
CREATE OR REPLACE FUNCTION TAdk.iban_validate__state(
    iban varchar
    )
  RETURNS varchar AS $$
  DECLARE
    char_value    integer;
    numeric_iban  varchar := '';
    shifted_iban  varchar;
    country_code  varchar;
    digit         varchar;
    remainder     integer := 0;
    iban_data     record;
  BEGIN
    -- Entferne Leerzeichen und konvertiere in Großbuchstaben
    iban := REPLACE(UPPER(iban), ' ', '');

    IF LENGTH(iban) < 15 OR LENGTH(iban) > 34 THEN
        RETURN coalesce( format( lang_text( 29971 ) ), 'Die eingegebene IBAN ist zu kurz oder zu lang' );
    END IF;

    -- Extrahiere den Ländercode
    country_code := SUBSTRING(iban FROM 1 FOR 2);

    -- Länderspezifische Längen- und Formatprüfung
    SELECT iban_expected_length, iban_regex
      INTO iban_data
      FROM laender_iban_formats
     WHERE iban_iso = country_code
     ORDER BY LENGTH(iban) = iban_expected_length DESC
     LIMIT 1;

    IF iban_data IS NULL then
        RETURN coalesce( format( lang_text( 29968 ), country_code ), 'Der Ländercode ' || country_code || ' wird nicht unterstützt oder ist ungültig' );
    END IF;

    -- Überprüfe die Länge für das spezifische Land
    IF LENGTH(iban) <> iban_data.iban_expected_length THEN
        RETURN coalesce( format( lang_text( 29967 ), country_code, iban_data.iban_expected_length), 'Die IBAN-Länge für ' || country_code || ' muss ' || iban_data.iban_expected_length || ' Zeichen betragen' );
    END IF;

    -- Überprüfe das Format mit Regex
    IF NOT iban ~ iban_data.iban_regex THEN
        --RETURN format( lang_text( 29969 ), country_code );  ---    'Das IBAN-Format für ' || country_code || ' ist ungültig';
        RETURN coalesce( format( lang_text( 29970 ) ), 'Das IBAN-Format für ' || country_code || ' ist ungültig' );
    END IF;

    -- Verschiebe die ersten vier Zeichen ans Ende
    shifted_iban := SUBSTRING(iban FROM 5) || SUBSTRING(iban FROM 1 FOR 4);

    -- Ersetze Buchstaben durch ihre numerischen Werte
    FOR i IN 1..LENGTH(shifted_iban) LOOP
        char_value := ASCII(SUBSTRING(shifted_iban FROM i FOR 1));
        IF char_value >= ASCII('A') AND char_value <= ASCII('Z') THEN
            numeric_iban := numeric_iban || (char_value - 55)::varchar;
        ELSE
            numeric_iban := numeric_iban || SUBSTRING(shifted_iban FROM i FOR 1);
        END IF;
    END LOOP;

    -- Führe Modulo 97 iterativ (aus Performance-Gründen) aus und prüfe auf 1
    FOR i IN 1..LENGTH(numeric_iban) LOOP
        digit := SUBSTRING(numeric_iban FROM i FOR 1);
        remainder := MOD(remainder * 10 + CAST(digit AS integer), 97);
    END LOOP;

    IF remainder <> 1 THEN
        RETURN coalesce( format( lang_text( 29970 ) ), 'Die IBAN hat eine falsche Prüfsumme' );
    ELSE
        RETURN '';
    END IF;

  END $$ LANGUAGE plpgsql STABLE;

---

CREATE OR REPLACE FUNCTION TAdk.iban_validate__bool( iban varchar ) RETURNS boolean AS $$
 DECLARE result varchar;
 BEGIN

    SELECT TAdk.iban_validate__state( iban ) INTO result;

    IF result <> '' THEN
        IF current_user = 'docker' THEN
            RAISE EXCEPTION '%', result;
        END IF;
        RETURN false;
    ELSE
        RETURN true;
    END IF;
 END $$ LANGUAGE plpgsql;
 --


 CREATE OR REPLACE FUNCTION tadk.leitweg_id__character__convert( _leitweg_char varchar )
  RETURNS varchar AS $$
  DECLARE
    _leitweg_char_lower_case   varchar;
    _result                    varchar;
  BEGIN

    -- es wird nur der erste Charakter von _leitweg_char berücksichtigt
    -- wandelt case-insensitiv Buchstaben in Zahlen-Strings um
    -- 'a','A' -> '10' .. 'z','Z' -> '35'
    -- alle anderen Zeichen bleiben unverändert

    _leitweg_char_lower_case := lower ( _leitweg_char );
    IF
        regexp_replace( _leitweg_char_lower_case, '[a-z]', '' ) = ''
    THEN
        _result := ascii( _leitweg_char_lower_case ) - ascii( 'a' ) + 10;
    ELSE
        _result := substring( _leitweg_char_lower_case, 1, 1);
    END IF;

    RETURN _result;
  END $$ LANGUAGE plpgsql IMMUTABLE;
  --

CREATE OR REPLACE FUNCTION tadk.leitweg_id__checksum__calculate( _leitweg_id_ohne_pruefziffer varchar )
  RETURNS varchar AS $$
  DECLARE

    -- Lietweg-ID ohne nicht-alphanumerische Zeichen
    _leitweg_id_alphanum           varchar;

    -- Zählvariable in Schleife
    _i                             integer;

    -- numerische Darstellung der Leitweg-ID als String
    _leitweg_id_numerisch_string   varchar;

    -- numerische Darstellung der Leitweg-ID als Zahl
    _leitweg_id_numerisch_integer  numeric;

    -- zweistellige Prüfziffer als String
    _pruefziffer_string            varchar;

    -- zweistellige Prüfziffer als Zahl
    _pruefziffer_integer           numeric;
  BEGIN

    -- errechnet eine Prüfziffer gemäß ISO/IEC 7064:2003 per Modulo 97-10 Verfahren
    -- _leitweg_id_ohne_pruef - Leitweg-ID ohne Prüfziffer-Suffix (z.B. '123-ABCDE')

    -- Bindestriche eliminieren
    _leitweg_id_alphanum := regexp_replace( _leitweg_id_ohne_pruefziffer, '-', '', E'\g');

    -- Leitweg-ID in numerische Repräsentation umwandeln
    -- case-insensitiv, 'a','A'->'10' .. 'z','Z'->'35', Zahlzeichen bleiben unverändert
    -- am Ende noch zwei Nullen für die Prüfziffer anhängen
    _leitweg_id_numerisch_string := '';
    FOR _i IN 1..length( _leitweg_id_alphanum ) LOOP
      _leitweg_id_numerisch_string := _leitweg_id_numerisch_string
        || tadk.leitweg_id__character__convert( substring( _leitweg_id_alphanum, _i, 1 ));
    END LOOP;
    _leitweg_id_numerisch_string := _leitweg_id_numerisch_string || '00';

    -- Prüfziffer ist im Wesentlichen diese numerische Repräsentation modulo 97
    _leitweg_id_numerisch_integer := _leitweg_id_numerisch_string :: numeric;
    _pruefziffer_integer  := 98 - _leitweg_id_numerisch_integer % 97;
    _pruefziffer_string   := _pruefziffer_integer :: varchar;

    IF length( _pruefziffer_string ) = 1 THEN _pruefziffer_string := '0' || _pruefziffer_string; END IF;
    RETURN _pruefziffer_string;
  END $$ LANGUAGE plpgsql IMMUTABLE STRICT;
  --

CREATE OR REPLACE FUNCTION tadk.leitweg_id__checksum__check( _leitweg_id varchar )
  RETURNS boolean AS $$
  DECLARE
    _position_letzter_bindestrich integer;
    _leitweg_id_ohne_pruefziffer varchar;
    _prueffer_uebergeben varchar;
    _pruef_berechnet varchar;
  BEGIN

    -- prüft, ob eine Leitweg-ID eine korrekte Prüfziffer enthält
    -- _leitweg_id - korrekt formatierte Leitweg-ID

    IF _leitweg_id IS null THEN RETURN true; END IF;

    _position_letzter_bindestrich := length( _leitweg_id ) - position( '-' in reverse( _leitweg_id )) + 1;
    _leitweg_id_ohne_pruefziffer  := substring( _leitweg_id, 1, _position_letzter_bindestrich - 1 );

    _prueffer_uebergeben := substring( _leitweg_id, _position_letzter_bindestrich + 1, 2 );
     _pruef_berechnet := tadk.leitweg_id__checksum__calculate( _leitweg_id_ohne_pruefziffer );

    RETURN _prueffer_uebergeben =  _pruef_berechnet AND  _pruef_berechnet IS NOT null;
  END $$ LANGUAGE plpgsql IMMUTABLE;
  --
--- #22533
CREATE OR REPLACE FUNCTION tsvector(_adk adk) RETURNS tsvector AS $$
 BEGIN
   RETURN to_tsvector( coalesce(_adk.ad_krz , '') || ' ' ||
                       coalesce(_adk.ad_fa1 , '') || ' ' ||
                       coalesce(_adk.ad_name, '') || ' ' ||
                       coalesce(_adk.ad_plz , '') || ' ' ||
                       coalesce(_adk.ad_ort , '') || ' ' ||
                       coalesce(_adk.ad_tel1, '')
                     );
 END;
 $$ LANGUAGE plpgsql IMMUTABLE;

CREATE INDEX idx_adk_search_vector ON adk USING GIN (tsvector(adk)); --DROP INDEX idx_adk_search_vector;
 ---